<?php

namespace Tests\Unit;

use App\Models\BankAccount;
use App\Models\BankTransaction;
use App\Models\Invoice;
use App\Models\Contact;
use App\Models\Tenant;
use App\Models\User;
use App\Services\BankingService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class BankingServiceTest extends TestCase
{
    use RefreshDatabase;

    protected BankingService $service;
    protected Tenant $tenant;
    protected BankAccount $account;

    protected function setUp(): void
    {
        parent::setUp();

        $this->service = app(BankingService::class);
        $this->tenant = Tenant::factory()->create();
        $this->account = BankAccount::factory()->create([
            'tenant_id' => $this->tenant->id,
            'current_balance' => 10000.000,
        ]);
    }

    public function test_imports_transactions(): void
    {
        $transactions = [
            [
                'date' => '2024-01-15',
                'description' => 'Payment from ABC Corp',
                'amount' => 500.000,
                'type' => 'credit',
                'reference' => 'TXN001',
            ],
            [
                'date' => '2024-01-16',
                'description' => 'Office supplies',
                'amount' => 50.000,
                'type' => 'debit',
                'reference' => 'TXN002',
            ],
        ];

        $imported = $this->service->importTransactions($this->account, $transactions);

        $this->assertEquals(2, $imported);
        $this->assertDatabaseHas('bank_transactions', ['reference' => 'TXN001']);
        $this->assertDatabaseHas('bank_transactions', ['reference' => 'TXN002']);
    }

    public function test_matches_transaction_to_invoice(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
            'total' => 500.000,
            'status' => 'sent',
        ]);

        $transaction = BankTransaction::factory()->create([
            'tenant_id' => $this->tenant->id,
            'bank_account_id' => $this->account->id,
            'amount' => 500.000,
            'type' => 'credit',
            'status' => 'unmatched',
        ]);

        $this->service->matchTransaction($transaction, 'invoice', $invoice->id);

        $this->assertEquals('matched', $transaction->fresh()->status);
        $this->assertEquals($invoice->id, $transaction->fresh()->matchable_id);
    }

    public function test_reconciles_account(): void
    {
        BankTransaction::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'bank_account_id' => $this->account->id,
            'status' => 'matched',
            'is_reconciled' => false,
        ]);

        $reconciliation = $this->service->reconcile($this->account, [
            'statement_date' => now()->toDateString(),
            'statement_balance' => 10000.000,
            'transaction_ids' => BankTransaction::pluck('id')->toArray(),
        ]);

        $this->assertEquals(5, $reconciliation->transactions_count);
        $this->assertTrue(BankTransaction::where('is_reconciled', true)->count() === 5);
    }

    public function test_suggests_matches(): void
    {
        $contact = Contact::factory()->create([
            'tenant_id' => $this->tenant->id,
            'company_name' => 'ABC Corporation',
        ]);
        
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
            'total' => 500.000,
            'status' => 'sent',
        ]);

        $transaction = BankTransaction::factory()->create([
            'tenant_id' => $this->tenant->id,
            'bank_account_id' => $this->account->id,
            'description' => 'Payment from ABC Corporation',
            'amount' => 500.000,
            'type' => 'credit',
            'status' => 'unmatched',
        ]);

        $suggestions = $this->service->suggestMatches($transaction);

        $this->assertNotEmpty($suggestions);
        $this->assertEquals($invoice->id, $suggestions[0]['id']);
    }
}
